// Admin Authentication System

// Initialize admin system
function initAdminAuth() {
    // Check if we're on admin pages
    const currentPath = window.location.pathname;
    const currentHref = window.location.href;
    
    // More comprehensive admin page detection
    const isAdminPage = currentPath.includes('/admin/') || 
                       currentPath.includes('\\admin\\') || 
                       currentPath.includes('/admin') ||
                       currentHref.includes('/admin/') ||
                       currentHref.includes('\\admin\\') ||
                       currentPath.toLowerCase().includes('admin');
    
    console.log('Current path:', currentPath);
    console.log('Current href:', currentHref);
    console.log('Is admin page:', isAdminPage);
    
    if (isAdminPage) {
        console.log('Initializing admin system...');
        // Check authentication for admin pages (except login)
        if (!currentPath.includes('login.html') && !isAdminLoggedIn()) {
            console.log('Redirecting to login...');
            window.location.href = 'login.html';
            return;
        }
        
        // Setup event listeners
        console.log('Setting up event listeners...');
        setupAdminEventListeners();
        
        // Load admin data
        if (currentPath.includes('dashboard.html') || currentHref.includes('dashboard.html')) {
            console.log('Loading admin dashboard...');
            loadAdminDashboard();
        }
    } else {
        console.log('Not an admin page, skipping admin initialization');
    }
}

// Check if admin is logged in
function isAdminLoggedIn() {
    return sessionStorage.getItem('adminUser') !== null;
}

// Get current admin user
function getCurrentAdmin() {
    const adminData = sessionStorage.getItem('adminUser');
    return adminData ? JSON.parse(adminData) : null;
}

// Setup event listeners
function setupAdminEventListeners() {
    // Admin login form
    const loginForm = document.getElementById('admin-login-form');
    if (loginForm) {
        loginForm.addEventListener('submit', handleAdminLogin);
    }
    
    // Admin logout
    const logoutBtn = document.getElementById('admin-logout');
    if (logoutBtn) {
        logoutBtn.addEventListener('click', handleAdminLogout);
    }
    
    // Sidebar toggle
    const sidebarToggle = document.getElementById('sidebar-toggle');
    if (sidebarToggle) {
        sidebarToggle.addEventListener('click', toggleSidebar);
    }
    
    // Profile dropdown
    const userProfile = document.getElementById('user-profile-menu');
    if (userProfile) {
        userProfile.addEventListener('click', toggleProfileDropdown);
    }
    
    // Close dropdown when clicking outside
    document.addEventListener('click', function(e) {
        const dropdown = document.getElementById('profile-dropdown');
        const userProfile = document.getElementById('user-profile-menu');
        
        if (dropdown && userProfile && 
            !dropdown.contains(e.target) && 
            !userProfile.contains(e.target)) {
            dropdown.classList.remove('show');
        }
    });
}

// Handle admin login
function handleAdminLogin(e) {
    e.preventDefault();
    
    const username = document.getElementById('admin-username').value.trim();
    const password = document.getElementById('admin-password').value;
    
    // Simple validation
    if (!username || !password) {
        showAdminNotification('Please fill in all fields', 'error');
        return;
    }
    
    // Check credentials (in a real app, this would be server-side)
    // For demo purposes, using hardcoded credentials
    const validCredentials = [
        { username: 'admin', password: 'admin123', role: 'Administrator' },
        { username: 'editor', password: 'editor123', role: 'Editor' },
        { username: 'viewer', password: 'viewer123', role: 'Viewer' }
    ];
    
    const user = validCredentials.find(cred => 
        cred.username === username && cred.password === password
    );
    
    if (user) {
        // Successful login
        const adminUser = {
            id: generateAdminId(),
            username: user.username,
            role: user.role,
            loginTime: new Date().toISOString(),
            lastActivity: new Date().toISOString()
        };
        
        sessionStorage.setItem('adminUser', JSON.stringify(adminUser));
        showAdminNotification('Login successful!', 'success');
        
        // Redirect to dashboard
        setTimeout(() => {
            window.location.href = 'dashboard.html';
        }, 1000);
    } else {
        showAdminNotification('Invalid username or password', 'error');
    }
}

// Handle admin logout
function handleAdminLogout(e) {
    e.preventDefault();
    
    if (confirm('Are you sure you want to logout?')) {
        sessionStorage.removeItem('adminUser');
        showAdminNotification('Logged out successfully', 'success');
        
        setTimeout(() => {
            window.location.href = 'login.html';
        }, 1000);
    }
}

// Toggle sidebar on mobile
function toggleSidebar() {
    const sidebar = document.querySelector('.sidebar');
    sidebar.classList.toggle('active');
}

// Toggle profile dropdown
function toggleProfileDropdown() {
    const dropdown = document.getElementById('profile-dropdown');
    dropdown.classList.toggle('show');
}

// Generate admin ID
function generateAdminId() {
    return 'admin_' + Date.now().toString(36) + Math.random().toString(36).substr(2);
}

// Show admin notification
function showAdminNotification(message, type = 'info') {
    // Remove existing notifications
    const existingNotifications = document.querySelectorAll('.admin-notification');
    existingNotifications.forEach(notification => notification.remove());
    
    const notification = document.createElement('div');
    notification.className = `admin-notification ${type}`;
    notification.textContent = message;
    
    // Add styling
    Object.assign(notification.style, {
        position: 'fixed',
        top: '20px',
        right: '20px',
        padding: '15px 20px',
        borderRadius: '8px',
        color: 'white',
        zIndex: '9999',
        maxWidth: '300px',
        boxShadow: '0 4px 6px rgba(0, 0, 0, 0.1)',
        fontWeight: '500'
    });
    
    // Set background color based on type
    switch(type) {
        case 'success':
            notification.style.backgroundColor = '#28a745';
            break;
        case 'error':
            notification.style.backgroundColor = '#dc3545';
            break;
        case 'warning':
            notification.style.backgroundColor = '#ffc107';
            notification.style.color = '#333';
            break;
        default:
            notification.style.backgroundColor = '#007bff';
    }
    
    document.body.appendChild(notification);
    
    // Auto remove after 3 seconds
    setTimeout(() => {
        if (notification.parentNode) {
            notification.remove();
        }
    }, 3000);
}

// Load admin dashboard data
function loadAdminDashboard() {
    if (!isAdminLoggedIn()) return;
    
    const adminUser = getCurrentAdmin();
    
    // Update admin name in header
    const adminNameElement = document.getElementById('admin-name');
    if (adminNameElement) {
        adminNameElement.textContent = adminUser.username;
    }
    
    // Load dashboard statistics
    loadDashboardStats();
    
    // Update last activity
    updateAdminActivity();
}

// Load dashboard statistics
function loadDashboardStats() {
    // In a real app, this would fetch from API
    // For demo, using sample data
    
    // Load users count
    const users = JSON.parse(localStorage.getItem('users') || '[]');
    document.getElementById('total-users').textContent = users.length;
    
    // Load products count
    const products = JSON.parse(localStorage.getItem('products') || '[]');
    document.getElementById('total-orders').textContent = products.length;
    
    // Sample data for other stats
    document.getElementById('total-pages').textContent = '12';
    document.getElementById('total-revenue').textContent = '₹45,230';
    document.getElementById('total-traffic').textContent = '1,248';
}

// Update admin activity
function updateAdminActivity() {
    const adminUser = getCurrentAdmin();
    if (adminUser) {
        adminUser.lastActivity = new Date().toISOString();
        sessionStorage.setItem('adminUser', JSON.stringify(adminUser));
    }
}

// Check admin role permissions
function hasPermission(requiredRole) {
    const adminUser = getCurrentAdmin();
    if (!adminUser) return false;
    
    const roleHierarchy = {
        'Viewer': 1,
        'Editor': 2,
        'Administrator': 3
    };
    
    const userRoleLevel = roleHierarchy[adminUser.role] || 0;
    const requiredRoleLevel = roleHierarchy[requiredRole] || 0;
    
    return userRoleLevel >= requiredRoleLevel;
}

// Initialize when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
    initAdminAuth();
    
    // Update activity on user interaction
    document.addEventListener('click', updateAdminActivity);
    document.addEventListener('keypress', updateAdminActivity);
});

// Handle window focus/blur for session management
window.addEventListener('focus', updateAdminActivity);
window.addEventListener('blur', function() {
    // Optional: Handle session timeout logic here
});