// Admin Dashboard Functionality

// Initialize dashboard
function initAdminDashboard() {
    if (!isAdminLoggedIn()) {
        window.location.href = 'login.html';
        return;
    }
    
    // Load dashboard data
    loadDashboardData();
    
    // Setup navigation
    setupDashboardNavigation();
    
    // Initialize charts (if needed)
    initCharts();
    
    // Update admin name display
    updateAdminNameDisplay();
}

// Update admin name display
function updateAdminNameDisplay() {
    const adminUser = getCurrentAdmin();
    if (adminUser) {
        const adminNameElement = document.getElementById('admin-name');
        if (adminNameElement) {
            adminNameElement.textContent = adminUser.username;
        }
    }
}

// Load dashboard data
function loadDashboardData() {
    // Load statistics
    loadStatistics();
    
    // Load recent activity
    loadRecentActivity();
    
    // Load quick actions based on user role
    loadQuickActions();
}

// Load statistics data
function loadStatistics() {
    // Users statistics
    const users = JSON.parse(localStorage.getItem('users') || '[]');
    const activeUsers = users.filter(user => {
        // Consider users active if they logged in within last 30 days
        const lastLogin = new Date(user.createdAt);
        const thirtyDaysAgo = new Date();
        thirtyDaysAgo.setDate(thirtyDaysAgo.getDate() - 30);
        return lastLogin > thirtyDaysAgo;
    });
    
    document.getElementById('total-users').textContent = users.length;
    
    // Products statistics
    const products = JSON.parse(localStorage.getItem('products') || '[]');
    const activeProducts = products.filter(product => !product.isSold);
    const soldProducts = products.filter(product => product.isSold);
    
    document.getElementById('total-orders').textContent = activeProducts.length;
    
    // Calculate sample revenue (assuming average product price)
    const totalRevenue = products.reduce((sum, product) => sum + (product.price || 0), 0);
    document.getElementById('total-revenue').textContent = formatCurrency(totalRevenue * 0.1); // 10% commission assumption
    
    // Sample traffic data
    document.getElementById('total-traffic').textContent = (users.length * 15 + products.length * 8);
    
    // Pages count (sample data)
    document.getElementById('total-pages').textContent = '8';
}

// Format currency
function formatCurrency(amount) {
    return new Intl.NumberFormat('en-IN', {
        style: 'currency',
        currency: 'INR',
        maximumFractionDigits: 0
    }).format(amount);
}

// Load recent activity
function loadRecentActivity() {
    const activityContainer = document.getElementById('recent-activity');
    if (!activityContainer) return;
    
    // Sample recent activities
    const recentActivities = [
        {
            type: 'user',
            icon: 'fa-user-plus',
            message: 'New user registered: <strong>Sarah Johnson</strong>',
            time: '5 minutes ago'
        },
        {
            type: 'product',
            icon: 'fa-box',
            message: 'New product added: <strong>MacBook Pro 2023</strong>',
            time: '12 minutes ago'
        },
        {
            type: 'comment',
            icon: 'fa-comment',
            message: 'New comment on blog post: <strong>Market Trends 2024</strong>',
            time: '24 minutes ago'
        },
        {
            type: 'order',
            icon: 'fa-shopping-cart',
            message: 'Product marked as sold: <strong>iPhone 14</strong>',
            time: '1 hour ago'
        },
        {
            type: 'user',
            icon: 'fa-user-edit',
            message: 'User profile updated: <strong>Mike Wilson</strong>',
            time: '2 hours ago'
        }
    ];
    
    // Clear existing content
    activityContainer.innerHTML = '';
    
    // Add activities to container
    recentActivities.forEach(activity => {
        const activityItem = document.createElement('div');
        activityItem.className = 'activity-item';
        activityItem.innerHTML = `
            <i class="fas ${activity.icon} activity-icon"></i>
            <div class="activity-content">
                <p>${activity.message}</p>
                <span class="activity-time">${activity.time}</span>
            </div>
        `;
        activityContainer.appendChild(activityItem);
    });
}

// Load quick actions based on user role
function loadQuickActions() {
    const adminUser = getCurrentAdmin();
    if (!adminUser) return;
    
    const quickActionsContainer = document.querySelector('.quick-actions');
    if (!quickActionsContainer) return;
    
    // Define actions based on role
    let actions = [];
    
    switch(adminUser.role) {
        case 'Administrator':
            actions = [
                { icon: 'fa-user-plus', text: 'Add User', action: 'users.html' },
                { icon: 'fa-plus-circle', text: 'Add Product', action: 'products.html' },
                { icon: 'fa-file', text: 'Create Page', action: 'pages.html' },
                { icon: 'fa-pen', text: 'Write Post', action: 'blog.html' }
            ];
            break;
            
        case 'Editor':
            actions = [
                { icon: 'fa-plus-circle', text: 'Add Product', action: 'products.html' },
                { icon: 'fa-file', text: 'Create Page', action: 'pages.html' },
                { icon: 'fa-pen', text: 'Write Post', action: 'blog.html' },
                { icon: 'fa-images', text: 'Media Manager', action: 'media.html' }
            ];
            break;
            
        case 'Viewer':
            actions = [
                { icon: 'fa-eye', text: 'View Reports', action: 'analytics.html' },
                { icon: 'fa-chart-line', text: 'Analytics', action: 'analytics.html' },
                { icon: 'fa-bell', text: 'Notifications', action: 'notifications.html' },
                { icon: 'fa-cog', text: 'Settings', action: 'settings.html' }
            ];
            break;
    }
    
    // Update quick actions
    quickActionsContainer.innerHTML = '';
    actions.forEach(action => {
        const actionBtn = document.createElement('button');
        actionBtn.className = 'action-btn';
        actionBtn.innerHTML = `
            <i class="fas ${action.icon}"></i>
            <span>${action.text}</span>
        `;
        actionBtn.addEventListener('click', () => {
            window.location.href = action.action;
        });
        quickActionsContainer.appendChild(actionBtn);
    });
}

// Setup dashboard navigation
function setupDashboardNavigation() {
    const navLinks = document.querySelectorAll('.nav-link');
    const currentPage = window.location.pathname.split('/').pop();
    
    navLinks.forEach(link => {
        link.classList.remove('active');
        if (link.getAttribute('href') === currentPage) {
            link.classList.add('active');
        }
    });
    
    // Update page title based on current page
    updatePageTitle();
}

// Update page title
function updatePageTitle() {
    const currentPage = window.location.pathname.split('/').pop();
    const pageTitles = {
        'dashboard.html': 'Dashboard Overview',
        'users.html': 'Users Management',
        'products.html': 'Products Management',
        'pages.html': 'Pages Management',
        'blog.html': 'Blog/Posts Management',
        'media.html': 'Media Manager',
        'seo.html': 'SEO Settings',
        'settings.html': 'Website Settings',
        'analytics.html': 'Analytics Dashboard',
        'notifications.html': 'Notifications',
        'security.html': 'Security Settings'
    };
    
    const titleElement = document.querySelector('.navbar-left h1');
    if (titleElement && pageTitles[currentPage]) {
        titleElement.textContent = pageTitles[currentPage];
    }
}

// Initialize charts (placeholder for future implementation)
function initCharts() {
    // This would integrate with charting libraries like Chart.js
    // For now, showing placeholder functionality
    
    console.log('Charts initialization ready');
}

// Update dashboard in real-time
function updateDashboardRealTime() {
    // Update statistics every 30 seconds
    setInterval(() => {
        if (document.visibilityState === 'visible') {
            loadStatistics();
            updateAdminActivity();
        }
    }, 30000);
}

// Handle window visibility changes
document.addEventListener('visibilitychange', function() {
    if (document.visibilityState === 'visible') {
        // Refresh data when tab becomes active
        loadDashboardData();
    }
});

// Manual initialization function (for debugging)
function manualInitDashboard() {
    console.log('Manual dashboard initialization');
    initAdminDashboard();
    updateDashboardRealTime();
}

// Make function available globally for testing
window.manualInitDashboard = manualInitDashboard;

// Initialize dashboard when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
    console.log('DOM loaded, checking if this is dashboard page...');
    
    // Only initialize if we're on dashboard page
    const currentPath = window.location.pathname;
    const currentHref = window.location.href;
    const isDashboardPage = currentPath.includes('dashboard.html') || 
                           currentPath.endsWith('dashboard.html') ||
                           currentHref.includes('dashboard.html') ||
                           currentPath.toLowerCase().includes('dashboard');
    
    console.log('Current path:', currentPath);
    console.log('Current href:', currentHref);
    console.log('Is dashboard page:', isDashboardPage);
    
    if (isDashboardPage) {
        console.log('Initializing dashboard...');
        // Small delay to ensure DOM is fully loaded
        setTimeout(() => {
            initAdminDashboard();
            updateDashboardRealTime();
        }, 100);
    } else {
        console.log('Not a dashboard page, skipping dashboard initialization');
    }
});