// Main application script for Direct'sell marketplace

// Initialize the application when DOM is loaded
document.addEventListener('DOMContentLoaded', function() {
    initializeApp();
});

// Initialize the entire application
function initializeApp() {
    // Initialize all modules
    initAuth();
    initProducts();
    
    // Set up general event listeners
    setupGeneralEventListeners();
    
    // Load specific content based on the current page
    loadPageContent();
    
    // Populate location filter
    populateLocationFilter();
}

// Set up general event listeners
function setupGeneralEventListeners() {
    // Mobile menu toggle
    const hamburger = document.querySelector('.hamburger');
    const navMenu = document.querySelector('.nav-menu');
    
    if (hamburger && navMenu) {
        hamburger.addEventListener('click', function() {
            navMenu.classList.toggle('active');
        });
    }
    
    // Close mobile menu when clicking on a link
    const navLinks = document.querySelectorAll('.nav-link');
    navLinks.forEach(link => {
        link.addEventListener('click', () => {
            if (navMenu) {
                navMenu.classList.remove('active');
            }
        });
    });
    
    // Update navigation on page load
    updateNavigation();
}

// Load content specific to the current page
function loadPageContent() {
    const currentPage = window.location.pathname.split('/').pop();
    
    switch(currentPage) {
        case 'profile.html':
            loadUserProfile();
            setupProfileTabs();
            break;
            
        case 'product-detail.html':
            loadProductDetail();
            break;
            
        case 'post-product.html':
            // Make sure user is logged in
            if (!requireAuth()) return;
            break;
            
        case 'login.html':
            // Set up tab functionality for login page
            setupLoginTabs();
            break;
            
        case 'index.html':
        default:
            // Load all products on homepage
            loadAllProducts();
            break;
    }
}

// Set up login tabs functionality
function setupLoginTabs() {
    // Check if we're on the login page
    if (window.location.pathname.includes('login.html')) {
        // Set up tab switching
        const tabButtons = document.querySelectorAll('.tab-btn');
        tabButtons.forEach(button => {
            button.addEventListener('click', switchAuthTab);
        });
        
        // If there are saved credentials, auto-fill login form
        const savedEmail = localStorage.getItem('rememberedEmail');
        if (savedEmail) {
            document.getElementById('login-email').value = savedEmail;
        }
    }
}

// Function to handle page transitions smoothly
function smoothTransition() {
    // Add fade-in effect when page loads
    document.body.style.opacity = '0';
    document.body.style.transition = 'opacity 0.3s ease-in-out';
    
    setTimeout(() => {
        document.body.style.opacity = '1';
    }, 100);
}

// Function to handle error boundaries
function handleError(error, context = '') {
    console.error(`Error in ${context}:`, error);
    showNotification('An error occurred. Please try again.', 'error');
}

// Function to handle offline/online status
function handleOnlineStatus() {
    window.addEventListener('online', () => {
        showNotification('You are back online', 'success');
    });
    
    window.addEventListener('offline', () => {
        showNotification('You are offline. Some features may not work.', 'warning');
    });
}

// Initialize service worker for offline functionality (if needed in future)
function initializeServiceWorker() {
    if ('serviceWorker' in navigator) {
        window.addEventListener('load', () => {
            // Service worker registration would go here in production
        });
    }
}

// Initialize analytics tracking (would be implemented in production)
function initializeAnalytics() {
    // Analytics code would go here
}

// Handle browser back/forward buttons
window.addEventListener('popstate', function(event) {
    // Handle navigation state changes
    loadPageContent();
});

// Initialize all functionality when page is fully loaded
window.addEventListener('load', function() {
    // Additional initialization after all resources are loaded
    smoothTransition();
    handleOnlineStatus();
    initializeServiceWorker();
    initializeAnalytics();
    
    // Update UI based on current login status
    updateNavigation();
});