// Utility functions for Direct'sell marketplace

// Generate unique IDs
function generateId() {
    return Date.now().toString(36) + Math.random().toString(36).substr(2);
}

// Format currency
function formatCurrency(amount) {
    return new Intl.NumberFormat('en-IN', {
        style: 'currency',
        currency: 'INR'
    }).format(amount);
}

// Format date
function formatDate(dateString) {
    const date = new Date(dateString);
    return date.toLocaleDateString('en-IN', {
        year: 'numeric',
        month: 'short',
        day: 'numeric'
    });
}

// Validate email
function isValidEmail(email) {
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    return emailRegex.test(email);
}

// Validate phone number (simple validation for Indian numbers)
function isValidPhone(phone) {
    const phoneRegex = /^(\+91[-\s]?)?[0]?(91)?[6789]\d{9}$/;
    return phoneRegex.test(phone);
}

// Show notification
function showNotification(message, type = 'info') {
    // Remove existing notifications
    const existingNotifications = document.querySelectorAll('.notification');
    existingNotifications.forEach(notification => notification.remove());

    const notification = document.createElement('div');
    notification.className = `notification ${type}`;
    notification.textContent = message;
    
    // Add styling
    Object.assign(notification.style, {
        position: 'fixed',
        top: '20px',
        right: '20px',
        padding: '15px 20px',
        borderRadius: '5px',
        color: 'white',
        zIndex: '9999',
        maxWidth: '300px',
        boxShadow: '0 4px 6px rgba(0, 0, 0, 0.1)'
    });

    // Set background color based on type
    switch(type) {
        case 'success':
            notification.style.backgroundColor = '#28a745';
            break;
        case 'error':
            notification.style.backgroundColor = '#dc3545';
            break;
        case 'warning':
            notification.style.backgroundColor = '#ffc107';
            notification.style.color = '#333';
            break;
        default:
            notification.style.backgroundColor = '#007bff';
    }

    document.body.appendChild(notification);

    // Auto remove after 3 seconds
    setTimeout(() => {
        if (notification.parentNode) {
            notification.remove();
        }
    }, 3000);
}

// Debounce function
function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

// Capitalize first letter of string
function capitalize(str) {
    return str.charAt(0).toUpperCase() + str.slice(1);
}

// Get category display name
function getCategoryDisplayName(category) {
    const categoryMap = {
        'garments': '👕 Garments',
        'shoes': '👟 Shoes',
        'electronics': '📱 Electronics',
        'home-items': '🏠 Home Items',
        'others': '🚗 Others'
    };
    return categoryMap[category] || capitalize(category.replace('-', ' '));
}

// Get condition display name
function getConditionDisplayName(condition) {
    const conditionMap = {
        'new': 'New',
        'used': 'Used'
    };
    return conditionMap[condition] || capitalize(condition);
}

// Format location
function formatLocation(location) {
    return location ? capitalize(location.toLowerCase()) : 'N/A';
}

// Check if user is logged in
function isLoggedIn() {
    return sessionStorage.getItem('currentUser') !== null;
}

// Get current user
function getCurrentUser() {
    const userData = sessionStorage.getItem('currentUser');
    if (userData) {
        return JSON.parse(userData);
    }
    return null;
}

// Save user to localStorage
function saveUser(user) {
    const users = JSON.parse(localStorage.getItem('users') || '[]');
    const existingIndex = users.findIndex(u => u.id === user.id);
    
    if (existingIndex > -1) {
        users[existingIndex] = user;
    } else {
        users.push(user);
    }
    
    localStorage.setItem('users', JSON.stringify(users));
}

// Load all users
function loadUsers() {
    return JSON.parse(localStorage.getItem('users') || '[]');
}

// Load all products
function loadProducts() {
    return JSON.parse(localStorage.getItem('products') || '[]');
}

// Save products to localStorage
function saveProducts(products) {
    localStorage.setItem('products', JSON.stringify(products));
}

// Get user by ID
function getUserById(userId) {
    const users = loadUsers();
    return users.find(user => user.id === userId);
}

// Update navigation based on login status
function updateNavigation() {
    const loginBtn = document.getElementById('login-btn');
    const logoutBtn = document.getElementById('logout-btn');
    const profileBtn = document.getElementById('profile-btn');
    const postProductBtn = document.getElementById('post-product-btn');

    if (isLoggedIn()) {
        if (loginBtn) loginBtn.style.display = 'none';
        if (logoutBtn) logoutBtn.style.display = 'block';
        if (profileBtn) profileBtn.href = 'profile.html';
        if (postProductBtn) postProductBtn.href = 'post-product.html';
    } else {
        if (loginBtn) loginBtn.style.display = 'block';
        if (logoutBtn) logoutBtn.style.display = 'none';
        if (profileBtn) profileBtn.href = 'login.html';
        if (postProductBtn) postProductBtn.href = 'login.html';
    }
}

// Update user in session storage
function updateCurrentUser(user) {
    sessionStorage.setItem('currentUser', JSON.stringify(user));
    updateNavigation();
}